/*
 * @(#)Connection.java  1.0  2. Dezember 2003
 *
 * Copyright (c) 2003 Lucerne University of Applied Sciences and Arts (HSLU)
 * Zentralstrasse 18, Postfach 2858, CH-6002 Lucerne, Switzerland
 * All rights reserved.
 *
 * The copyright of this software is owned by the Lucerne University of Applied 
 * Sciences and Arts (HSLU). You may not use, copy or modify this software, 
 * except in accordance with the license agreement you entered into with HSLU. 
 * For details see accompanying license terms. 
 */


package ch.hslu.cm.simulation;

import java.util.*;
/**
 * A Relationship connects two Elements.
 *
 * @author  Werner Randelshofer
 * @version 1.0 2. Dezember 2003  Created.
 */
public interface SimulatedRelationship extends SimulatedObject {
    /**
     * Sets the start element of the relationship and adds the relationship
     * to the simulated element.
     * If the start point and the end point connect to the same simulated
     * element, the relationship is not added to the simulated element.
     */
    public void setStart(SimulatedElement start);
    /**
     * Sets the end element of the relationship and adds the relationship
     * to the simulated element.
     * If the start point and the end point connect to the same simulated
     * element, the relationship is not added to the simulated element.
     */
    public void setEnd(SimulatedElement end);
    
    /**
     * Gets the element at the start of the relationship.
     */
    public SimulatedElement getStart();
    /**
     * Gets the element at the end of the relationship.
     */
    public SimulatedElement getEnd();
    
    /**
     * Returns the connected element at the other end of the relationship.
     *
     * @param startOrEnd the element at the start or the end of the relationship.
     */
    public SimulatedElement getConnected(SimulatedElement startOrEnd);
    
    /**
     * Returns true if this relationship can connect the provided start and end
     * elements.
     */
    public boolean canConnect(SimulatedElement start, SimulatedElement end);

    /**
     * Returns true if this relationship can connect the provided start element.
     */
    public boolean canConnect(SimulatedElement start);
    
    /**
     * After cloning a collection of elements, the Connection contained
     * in this collection still connect to the original elements instead of
     * to the clones.
     * Using this method and providing a map, which maps from the original
     * collection of elements to the new collection, this relationship remaps
     * to the new elements.
     */
    public void remap(Map oldToNew);

    /**
     * Activates this relationship.
     * A relationship may be activated multiple times.
     */
    public void activate();
    /**
     * Deactivates this relationship.
     * A relationship may be deactivated as many times as it has been activated.
     */
    public void deactivate();
    /**
     * Sets the growth of the relationship.
     *
     * @param percentage A percentage ranging from 0 to 100. 0 equals to no
     * growth, 100 equals to a fully grown relationship.
     */
    public void setGrowth(int percentage);
    
    /**
     * Returns true if this relationship is active.
     */
    public boolean isActive();
    
    /**
     * Returns the growth of the relationship.
     *
     * @return A percentage ranging from 0 to 100. 0 equals to no
     * growth, 100 equals to a fully grown relationship.
     */
    public int getGrowth();
    
    /**
     * Let's the relationship "grow" from the start to the end of the relationship.
     */
    public void grow();
    /**
     * Let's the relationship "shrink" from the end to the start of the relationship.
     */
    public void shrink();
}
